<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Carbon\Carbon;

class Dossier extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'insured_id',
        'opponent_id',
        'company_id',
        'garage_id',
        'guarantee_id',
        'intermidiaire_id',
        'type_dossier_id',
        'type_expertise_id',
        'dossier_status_id',
        'vehicule_model_id',
        'numero_sinistre',
        'numero_police',
        'numero_attestation',
        'numero_chassis',
        'sinistre_douteux',
        'date_sinistre',
        'date_reception',
        'ref',
        'new_matricule',
        'matricule',
        'mode_id',
        'kilometrage',
        'puissance_fiscale',
        'date_circulation',
        'step_id',
        'date_fin',
        'carburant_id',
        'valeurneuf',
        'valeuradire',
        'horssinistre',
        'valeurepave',
        'valeurvenale',
        'valeurneuf_ttc',
        'valeuradire_ttc',
        'horssinistre_ttc',
        'valeurepave_ttc',
        'valeurvenale_ttc',
        'user_id',
        'type_vehicule_id',
        'from',
        'to',
        'city_id',
        'currency_id',
        'facture',
        'date_facturation',
        'montant_accord_adverse',
        'update'
    ];

    protected $casts = [
        // Date casts removed - dates will return as strings like before
        // 'date_sinistre' => 'date',
        // 'date_reception' => 'date',
        // 'date_circulation' => 'date',
        // 'date_fin' => 'date',
        // 'date_facturation' => 'date',
        'sinistre_douteux' => 'boolean',
        'update' => 'boolean',
    ];

    // Append computed attributes only when needed
    protected $appends = ['timeleft'];

    /**
     * =====================================================
     * RELATIONSHIPS - Full columns for edit forms
     * =====================================================
     */

    // REMOVED select() from all relationships to prevent data loss
    // For dashboard optimization, use withDashboardRelations() scope instead

    public function currency()
    {
        return $this->belongsTo(Currency::class);
    }

    public function insured()
    {
        return $this->belongsTo(Insured::class);
    }

    public function city()
    {
        return $this->belongsTo(City::class);
    }

    public function carburant()
    {
        return $this->belongsTo(Carburant::class);
    }

    public function typevehicule()
    {
        return $this->belongsTo(TypeVehicule::class, 'type_vehicule_id');
    }

    public function opponent()
    {
        return $this->belongsTo(Opponent::class);
    }

    public function guarantee()
    {
        return $this->belongsTo(Guarantee::class);
    }

    public function document()
    {
        return $this->hasMany(Document::class);
    }

    public function piecetype()
    {
        return $this->belongsToMany(PieceType::class);
    }

    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    public function mode()
    {
        return $this->belongsTo(Mode::class);
    }

    public function typeexpertise()
    {
        return $this->belongsTo(TypeExpertise::class, 'type_expertise_id');
    }

    public function status()
    {
        return $this->belongsTo(DossierStatus::class, 'dossier_status_id');
    }

    public function dossierstatus()
    {
        return $this->belongsToMany(DossierStatus::class, 'dossier_status')->withTimestamps();
    }

    public function observation()
    {
        return $this->hasMany(Observation::class);
    }

    public function transaction()
    {
        return $this->hasMany(Transaction::class);
    }

    public function meeting()
    {
        return $this->hasMany(Meeting::class);
    }

    public function onemeeting()
    {
        return $this->hasOne(Meeting::class)->latest();
    }

    public function onedocument()
    {
        return $this->hasOne(Document::class)->where('type_document_id', 5);
    }

    public function contradictoire()
    {
        return $this->hasOne(Contradictoire::class);
    }

    public function step()
    {
        return $this->belongsToMany(Step::class, 'step_dossier')
            ->withPivot('user_id')
            ->withTimestamps();
    }

    public function shockpoint()
    {
        return $this->belongsToMany(ShockPoint::class, 'dossier_shockpoint');
    }

    public function devis()
    {
        return $this->hasMany(Devis::class);
    }

    public function onedevis()
    {
        return $this->hasOne(Devis::class)->latest();
    }

    public function notification()
    {
        return $this->hasMany(Notification::class);
    }

    public function vehiculemodel()
    {
        return $this->belongsTo(VehiculeModel::class, 'vehicule_model_id');
    }

    public function demande()
    {
        return $this->hasMany(Demande::class);
    }

    public function intermidiaire()
    {
        return $this->belongsTo(Intermidiaire::class);
    }

    public function garage()
    {
        return $this->belongsTo(Garage::class);
    }

    public function rapport()
    {
        return $this->hasMany(Rapport::class);
    }

    public function onerapport()
    {
        return $this->hasOne(Rapport::class)->latest();
    }

    public function facture()
    {
        return $this->hasMany(Facture::class);
    }

    public function onefacture()
    {
        return $this->hasOne(Facture::class)->latest();
    }

    public function lastmeet()
    {
        return $this->hasOne(Meeting::class)->latest();
    }

    public function montant()
    {
        return $this->hasOne(Montant::class);
    }

    public function payment()
    {
        return $this->hasOne(Payment::class);
    }

    public function paymentArbitrage()
    {
        return $this->hasMany(Payment::class);
    }

    public function signedurl()
    {
        return $this->hasMany(SignedUrl::class);
    }

    public function currentStep()
    {
        return $this->belongsTo(Step::class, 'step_id');
    }

    public function lastdevis()
    {
        return $this->hasOne(Devis::class)
            ->where('devis_status_id', 4)
            ->latest();
    }

    /**
     * Relationship to step_dossier pivot for optimized timeleft calculation
     */
    public function stepDates()
    {
        return $this->hasMany(StepDossier::class, 'dossier_id')
            ->select('dossier_id', 'step_id', 'created_at')
            ->latest();
    }

    /**
     * =====================================================
     * QUERY SCOPES - Optimized for Performance
     * =====================================================
     */

    /**
     * Scope for active dossiers (status 2, not updated)
     */
    public function scopeActive($query)
    {
        return $query->where('dossier_status_id', 2)
            ->where('update', 0);
    }

    /**
     * Scope to eager load common dashboard relations with selective columns
     * USE THIS in dashboard queries for performance
     */
    public function scopeWithDashboardRelations($query)
    {
        return $query->select([
                'dossiers.id',
                'dossiers.ref',
                'dossiers.company_id',
                'dossiers.insured_id',
                'dossiers.matricule',
                'dossiers.new_matricule',
                'dossiers.numero_sinistre',
                'dossiers.type_expertise_id',
                'dossiers.mode_id',
                'dossiers.step_id',
                'dossiers.dossier_status_id',
                'dossiers.update',
                'dossiers.sinistre_douteux',
                'dossiers.user_id',
                'dossiers.garage_id',
                'dossiers.valeurvenale',
                'dossiers.valeurepave',
                'dossiers.created_at',
                'dossiers.updated_at'
            ])
            ->with([
                'company' => function($query) {
                    $query->select('id', 'company');
                },
                'insured' => function($query) {
                    $query->select('id', 'name');
                },
                'typeexpertise' => function($query) {
                    $query->select('id', 'type');
                },
                'mode' => function($query) {
                    $query->select('id', 'mode');
                },
                'status' => function($query) {
                    $query->select('id', 'status', 'color');
                },
                'currentStep' => function($query) {
                    $query->select('id', 'step');
                },
                'onedevis' => function($query) {
                    $query->select('id', 'dossier_id', 'montant_devis', 'devis_status_id', 'created_at');
                },
                'onefacture' => function($query) {
                    $query->select('id', 'dossier_id', 'montant_facture', 'devis_status_id', 'created_at');
                },
                'onerapport' => function($query) {
                    $query->select('id', 'dossier_id', 'rapport_status_id', 'created_at');
                },
                'montant' => function($query) {
                    $query->select('id', 'dossier_id', 'montant_accord');
                },
            ]);
    }

    /**
     * Optimized User scope - applies user-specific filters
     */
    public function scopeUser($query)
    {
        $user = Auth::user();
        
        // Full access roles
        if ($user->role_id == 4 || Gate::allows('fullaccess', $this)) {
            return $query;
        }

        // Role 6 - Garage users
        if ($user->role_id == 6) {
            $garageIds = $user->manygarage()->pluck('garages.id');
            return $query->whereIn('garage_id', $garageIds);
        }

        // Get user company types once
        $userCompanyTypes = $user->userCompanyType;

        if ($userCompanyTypes->isEmpty()) {
            return $query->whereRaw('1 = 0'); // Return empty result
        }

        // Build the query for multiple company types
        $query->where(function ($subQuery) use ($userCompanyTypes) {
            foreach ($userCompanyTypes as $userCompanyType) {
                $subQuery->orWhere(function ($q) use ($userCompanyType) {
                    $q->where('company_id', $userCompanyType->pivot->company_id)
                      ->where('type_expertise_id', $userCompanyType->pivot->type_expertise_id);
                });
            }
        });

        return $query;
    }

    /**
     * Scope for filtering by devis status
     */
    public function scopeWithDevisStatus($query, $statusId)
    {
        return $query->whereHas('onedevis', function ($q) use ($statusId) {
            $q->where('devis_status_id', $statusId);
        });
    }

    /**
     * Scope for filtering by multiple devis statuses
     */
    public function scopeWithDevisStatuses($query, array $statusIds)
    {
        return $query->whereHas('onedevis', function ($q) use ($statusIds) {
            $q->whereIn('devis_status_id', $statusIds);
        });
    }

    /**
     * Scope for filtering by facture status
     */
    public function scopeWithFactureStatus($query, $statusId)
    {
        return $query->whereHas('onefacture', function ($q) use ($statusId) {
            $q->where('devis_status_id', $statusId);
        });
    }

    /**
     * Scope for filtering by multiple facture statuses
     */
    public function scopeWithFactureStatuses($query, array $statusIds)
    {
        return $query->whereHas('onefacture', function ($q) use ($statusIds) {
            $q->whereIn('devis_status_id', $statusIds);
        });
    }

    /**
     * Scope for filtering by rapport status
     */
    public function scopeWithRapportStatus($query, $statusId)
    {
        return $query->whereHas('onerapport', function ($q) use ($statusId) {
            $q->where('rapport_status_id', $statusId);
        });
    }

    /**
     * Scope for filtering by multiple rapport statuses
     */
    public function scopeWithRapportStatuses($query, array $statusIds)
    {
        return $query->whereHas('onerapport', function ($q) use ($statusIds) {
            $q->whereIn('rapport_status_id', $statusIds);
        });
    }

    /**
     * Scope for dossiers without devis
     */
    public function scopeWithoutDevis($query)
    {
        return $query->whereDoesntHave('devis');
    }

    /**
     * Scope for dossiers without facture
     */
    public function scopeWithoutFacture($query)
    {
        return $query->whereDoesntHave('facture');
    }

    /**
     * Scope for dossiers without rapport
     */
    public function scopeWithoutRapport($query)
    {
        return $query->whereDoesntHave('rapport');
    }

    /**
     * Scope for dossiers without specific document type
     */
    public function scopeWithoutDocument($query, $typeDocumentId)
    {
        return $query->whereDoesntHave('document', function ($q) use ($typeDocumentId) {
            $q->where('type_document_id', $typeDocumentId);
        });
    }

    /**
     * Scope for dossiers with specific document type
     */
    public function scopeWithDocument($query, $typeDocumentId)
    {
        return $query->whereHas('document', function ($q) use ($typeDocumentId) {
            $q->where('type_document_id', $typeDocumentId);
        });
    }

    /**
     * Scope for tribunal dossiers
     */
    public function scopeTribunale($query)
    {
        return $query->where('type_expertise_id', 9);
    }

    /**
     * Scope for suspicious dossiers
     */
    public function scopeDouteux($query)
    {
        return $query->where('sinistre_douteux', 1);
    }

    /**
     * Scope for rapid dossiers
     */
    public function scopeRapide($query, $roleId = null)
    {
        $roleId = $roleId ?? Auth::user()->role_id;
        
        $query->where('type_expertise_id', 16);
        
        if ($roleId == 2) {
            $query->whereIn('step_id', [4, 9]);
        }
        
        return $query;
    }

    /**
     * Scope for documents awaiting signature
     */
    public function scopeWithDocumentsToSign($query, $typeDocumentId = 21)
    {
        return $query->whereHas('document', function ($q) use ($typeDocumentId) {
            $q->where('type_document_id', $typeDocumentId)
              ->where('signer', 0)
              ->where('rejected', 0);
        });
    }

    /**
     * Scope for signed documents (without sent/validated rapport)
     */
    public function scopeWithSignedDocuments($query, $typeDocumentId = 21)
    {
        return $query->whereHas('document', function ($q) use ($typeDocumentId) {
            $q->where('type_document_id', $typeDocumentId)
              ->where('signer', 1)
              ->where('rejected', 0);
        })->whereDoesntHave('onerapport', function ($q) {
            $q->whereIn('rapport_status_id', [2, 3]);
        });
    }

    /**
     * Scope for base documents missing (documents 1, 2, 3)
     */
    public function scopeWithoutBaseDocuments($query)
    {
        return $query->whereDoesntHave('document', function ($q) {
            $q->whereIn('type_document_id', [1, 2, 3]);
        })->distinct();
    }

    /**
     * Scope for dossiers with meeting status
     */
    public function scopeWithMeetingStatus($query, $statusId)
    {
        return $query->whereHas('meeting', function ($q) use ($statusId) {
            $q->where('meeting_status_id', $statusId);
        });
    }

    /**
     * Scope for dossiers with latest meeting status
     */
    public function scopeWithLatestMeetingStatus($query, $statusId)
    {
        return $query->whereHas('onemeeting', function ($q) use ($statusId) {
            $q->where('meeting_status_id', $statusId);
        });
    }

    /**
     * Scope for dossiers needing expertise approval photo
     */
    public function scopeNeedingExpertiseApprovalPhoto($query)
    {
        return $query->where('type_expertise_id', '!=', 14)
            ->where('mode_id', '!=', 7)
            ->whereDoesntHave('document', function ($q) {
                $q->where('type_document_id', 6);
            })
            ->whereHas('onedevis', function ($q) {
                $q->whereIn('devis_status_id', [2, 7]);
            })
            ->whereDoesntHave('facture');
    }

    /**
     * Scope to apply seuil filter for role 2 users
     */
    public function scopeWithinSeuil($query, $userId = null, $type = 'both')
    {
        $userId = $userId ?? Auth::id();
        
        return $query->where(function ($q) use ($userId, $type) {
            $q->whereExists(function ($subQuery) use ($userId, $type) {
                $subQuery->select(DB::raw(1))
                    ->from('seuils')
                    ->whereColumn('seuils.company_id', 'dossiers.company_id')
                    ->where('seuils.user_id', $userId)
                    ->where(function ($sq) use ($type) {
                        if ($type === 'devis' || $type === 'both') {
                            $sq->whereExists(function ($devisQuery) {
                                $devisQuery->select(DB::raw(1))
                                    ->from('devis')
                                    ->whereColumn('devis.dossier_id', 'dossiers.id')
                                    ->whereRaw('devis.montant_devis > seuils.min')
                                    ->whereRaw('devis.montant_devis < seuils.max')
                                    ->orderBy('created_at', 'desc')
                                    ->limit(1);
                            });
                        }
                        
                        if ($type === 'facture' || $type === 'both') {
                            $method = ($type === 'both') ? 'orWhereExists' : 'whereExists';
                            $sq->$method(function ($factureQuery) {
                                $factureQuery->select(DB::raw(1))
                                    ->from('factures')
                                    ->whereColumn('factures.dossier_id', 'dossiers.id')
                                    ->whereRaw('factures.montant_facture > seuils.min')
                                    ->whereRaw('factures.montant_facture < seuils.max')
                                    ->orderBy('created_at', 'desc')
                                    ->limit(1);
                            });
                        }
                    });
            });
        });
    }

    /**
     * Scope for control devis (accord below seuil)
     */
    public function scopeControlDevis($query, $userId = null)
    {
        $userId = $userId ?? Auth::id();
        
        return $query->whereExists(function ($subQuery) use ($userId) {
            $subQuery->select(DB::raw(1))
                ->from('seuils')
                ->join('montants', 'montants.dossier_id', '=', 'dossiers.id')
                ->whereColumn('seuils.company_id', 'dossiers.company_id')
                ->where('seuils.user_id', $userId)
                ->whereRaw('montants.montant_accord < seuils.accord');
        });
    }

    /**
     * Scope for dossiers that need VV/VE values (mode 2 or 3)
     */
    public function scopeNeedingVvVe($query)
    {
        return $query->whereIn('mode_id', [2, 3])
            ->where(function ($q) {
                $q->whereNull('valeurvenale')
                  ->orWhereNull('valeurepave');
            });
    }

    /**
     * =====================================================
     * ACCESSORS - Computed Attributes
     * =====================================================
     */

    /**
     * Get the time left for current step
     * This is expensive, only compute when explicitly requested
     */
    public function getTimeleftAttribute()
    {
        // Return cached value if already computed
        if (isset($this->attributes['_timeleft_cache'])) {
            return $this->attributes['_timeleft_cache'];
        }

        $timeleft = $this->calculateTimeleft();
        $this->attributes['_timeleft_cache'] = $timeleft;
        
        return $timeleft;
    }

    /**
     * Calculate time left for this dossier
     * Separated method for easier testing and reuse
     */
    public function calculateTimeleft()
    {
        if (!$this->company_id || !$this->step_id) {
            return 0;
        }

        try {
            // Get company with step limit
            $companyStep = DB::table('company_step')
                ->where('company_id', $this->company_id)
                ->where('step_id', $this->step_id)
                ->first();

            if (!$companyStep || !$companyStep->time) {
                return 0;
            }

            // Get latest step date
            $stepDate = DB::table('step_dossier')
                ->where('dossier_id', $this->id)
                ->where('step_id', $this->step_id)
                ->latest('created_at')
                ->value('created_at');

            if (!$stepDate) {
                return 0;
            }

            $hoursPassed = Carbon::now()->diffInHours($stepDate);
            $limit = $companyStep->time;

            return $limit - $hoursPassed;
        } catch (\Exception $e) {
            \Log::error("Error calculating timeleft for dossier {$this->id}: " . $e->getMessage());
            return 0;
        }
    }

    /**
     * Check if dossier is overdue
     */
    public function getIsOverdueAttribute()
    {
        return $this->timeleft < 0;
    }

    /**
     * Get matricule (prioritize matricule over new_matricule)
     */
    public function getMatriculeDisplayAttribute()
    {
        return $this->matricule ?? $this->new_matricule;
    }

    /**
     * =====================================================
     * HELPER METHODS
     * =====================================================
     */

    /**
     * Check if dossier has base documents
     */
    public function hasBaseDocuments()
    {
        return $this->document()
            ->whereIn('type_document_id', [1, 2, 3])
            ->exists();
    }

    /**
     * Get the latest devis for this dossier
     */
    public function getLatestDevis()
    {
        return $this->onedevis;
    }

    /**
     * Get the latest facture for this dossier
     */
    public function getLatestFacture()
    {
        return $this->onefacture;
    }

    /**
     * Get the latest rapport for this dossier
     */
    public function getLatestRapport()
    {
        return $this->onerapport;
    }

    /**
     * Check if user can view this dossier
     */
    public function canView($user = null)
    {
        $user = $user ?? Auth::user();
        
        if ($user->role_id == 4 || Gate::allows('fullaccess', $this)) {
            return true;
        }

        if ($user->role_id == 6) {
            return $user->manygarage()->pluck('garages.id')->contains($this->garage_id);
        }

        $userCompanyTypes = $user->userCompanyType;
        
        foreach ($userCompanyTypes as $userCompanyType) {
            if ($this->company_id == $userCompanyType->pivot->company_id 
                && $this->type_expertise_id == $userCompanyType->pivot->type_expertise_id) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check if dossier is in specific mode
     */
    public function isMode($modeId)
    {
        return $this->mode_id == $modeId;
    }

    /**
     * Check if dossier has VV or VE values
     */
    public function hasVvOrVe()
    {
        return !is_null($this->valeurvenale) || !is_null($this->valeurepave);
    }

    /**
     * Get status color for display
     */
    public function getStatusColor()
    {
        return $this->status ? $this->status->color : '#000000';
    }

    /**
     * Get formatted date sinistre
     */
    public function getFormattedDateSinistre()
    {
        return $this->date_sinistre ? $this->date_sinistre->format('d/m/Y') : null;
    }

    /**
     * Get formatted date reception
     */
    public function getFormattedDateReception()
    {
        return $this->date_reception ? $this->date_reception->format('d/m/Y') : null;
    }
}