<?php

namespace App\Http\Controllers;

use App\Models\Company;
use App\Models\Dashboard;
use App\Models\Demande;
use App\Models\Dossier;
use App\Models\Rendezvous;
use App\Models\Rapport;
use App\Models\Seuil;
use App\Models\Ticket;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    /**
     * Cache duration in minutes
     */
    private const CACHE_DURATION = 5;

    /**
     * Default limit for queries
     */
    private const DEFAULT_LIMIT = 100;

    /**
     * =====================================================
     * BASE HELPER METHODS
     * =====================================================
     */

    /**
     * Get base query for dossiers with all necessary relations
     */
    private function getBaseDossiersQuery()
    {
        return Dossier::active()
            ->user()
            ->withDashboardRelations();
    }

    /**
     * Attach timeleft to dossiers collection efficiently
     * Calculates timeleft for all dossiers in just 2 queries
     */
    private function attachTimeleft($dossiers)
    {
        if ($dossiers->isEmpty()) {
            return $dossiers;
        }

        $dossierIds = $dossiers->pluck('id')->toArray();
        $companyIds = $dossiers->pluck('company_id')->unique()->filter()->toArray();

        if (empty($companyIds)) {
            $dossiers->each(fn($d) => $d->timeleft = 0);
            return $dossiers;
        }

        // Get all step dates in one query
        $stepDates = DB::table('step_dossier')
            ->whereIn('dossier_id', $dossierIds)
            ->select('dossier_id', 'step_id', 'created_at')
            ->orderBy('created_at', 'desc')
            ->get()
            ->groupBy('dossier_id')
            ->map(fn($group) => $group->groupBy('step_id')->map->first());

        // Get all company step limits in one query
        $companySteps = DB::table('company_step')
            ->whereIn('company_id', $companyIds)
            ->select('company_id', 'step_id', 'time')
            ->get()
            ->groupBy('company_id')
            ->map(fn($group) => $group->keyBy('step_id'));

        $now = now();

        // Calculate timeleft for each dossier
        foreach ($dossiers as $dossier) {
            $timeleft = 0;

            if ($dossier->company_id 
                && $dossier->step_id 
                && isset($companySteps[$dossier->company_id])
                && isset($companySteps[$dossier->company_id][$dossier->step_id])
                && isset($stepDates[$dossier->id])
                && isset($stepDates[$dossier->id][$dossier->step_id])
            ) {
                $stepDate = $stepDates[$dossier->id][$dossier->step_id];
                $limit = $companySteps[$dossier->company_id][$dossier->step_id]->time ?? 0;

                if ($limit > 0 && $stepDate) {
                    $hoursPassed = $now->diffInHours($stepDate->created_at);
                    $timeleft = $limit - $hoursPassed;
                }
            }

            $dossier->timeleft = $timeleft;
        }

        return $dossiers;
    }
    /**
     * =====================================================
     * MAIN DASHBOARD METHOD
     * =====================================================
     */

    public function front()
    {
        $userId = Auth::id();
        $roleId = Auth::user()->role_id;
        $cacheKey = "dashboard_front_{$userId}_{$roleId}";

        $data = Cache::remember($cacheKey, now()->addMinutes(self::CACHE_DURATION), function () use ($roleId) {
            // Get initial dossiers
            $dossiers = $this->getBaseDossiersQuery()
                ->latest()
                ->limit(self::DEFAULT_LIMIT)
                ->get();

            $this->attachTimeleft($dossiers);

            // Get all counts efficiently
            $counts = $this->getAllDashboardCounts();

            // Get ticket counts
            $ticketcount = Ticket::where('user_id', Auth::id())
                ->where('status_rendezvous_id', 2)
                ->count();

            $ticketcountterminer = Ticket::where('creator_id', Auth::id())
                ->where('status_rendezvous_id', 3)
                ->count();

            return compact('dossiers', 'ticketcount', 'ticketcountterminer') + $counts;
        });

        return view('dashboards.front', $data);
    }

    /**
     * Get all dashboard counts efficiently using batch queries
     * Optimized to minimize database round trips
     */
    private function getAllDashboardCounts()
    {
        $userId = Auth::id();
        $roleId = Auth::user()->role_id;

        // Batch query 1: Get all basic dossier counts in ONE query
        $dossierCounts = DB::table('dossiers')
            ->selectRaw("
                COUNT(CASE WHEN type_expertise_id = 9 AND dossier_status_id = 2 AND `update` = 0 THEN 1 END) as tribunale,
                COUNT(CASE WHEN sinistre_douteux = 1 AND dossier_status_id = 2 AND `update` = 0 THEN 1 END) as douteux,
                COUNT(CASE WHEN type_expertise_id = 16 AND dossier_status_id = 2 AND `update` = 0 " . 
                    ($roleId == 2 ? "AND step_id IN (4, 9)" : "") . " THEN 1 END) as rapide,
                COUNT(CASE WHEN dossier_status_id = 2 AND `update` = 0 AND id NOT IN (
                    SELECT dossier_id FROM devis
                ) THEN 1 END) as attdevis,
                COUNT(CASE WHEN dossier_status_id = 2 AND `update` = 0 AND id NOT IN (
                    SELECT dossier_id FROM factures
                ) THEN 1 END) as attfacture_base,
                COUNT(CASE WHEN dossier_status_id = 2 AND `update` = 0 AND id NOT IN (
                    SELECT dossier_id FROM rapports
                ) THEN 1 END) as attrapport_base,
                COUNT(CASE WHEN facture IS NULL AND dossier_status_id = 2 AND `update` = 1 THEN 1 END) as dossiersfacture
            ")
            ->where('dossier_status_id', 2)
            ->where('update', 0)
            ->first();

        // Batch query 2: Document counts in ONE query
        $docCounts = DB::table('dossiers')
            ->selectRaw("
                COUNT(DISTINCT CASE WHEN EXISTS (
                    SELECT 1 FROM documents 
                    WHERE documents.dossier_id = dossiers.id 
                    AND documents.type_document_id = 21 
                    AND documents.signer = 0 
                    AND documents.rejected = 0
                ) THEN dossiers.id END) as docasigner,
                COUNT(DISTINCT CASE WHEN EXISTS (
                    SELECT 1 FROM documents 
                    WHERE documents.dossier_id = dossiers.id 
                    AND documents.type_document_id = 21 
                    AND documents.signer = 1 
                    AND documents.rejected = 0
                ) AND NOT EXISTS (
                    SELECT 1 FROM rapports 
                    WHERE rapports.dossier_id = dossiers.id 
                    AND rapports.rapport_status_id IN (2, 3)
                    ORDER BY rapports.created_at DESC LIMIT 1
                ) THEN dossiers.id END) as docsigner,
                COUNT(DISTINCT CASE WHEN NOT EXISTS (
                    SELECT 1 FROM documents 
                    WHERE documents.dossier_id = dossiers.id 
                    AND documents.type_document_id IN (1, 2, 3)
                ) THEN dossiers.id END) as basedoc,
                COUNT(DISTINCT CASE WHEN EXISTS (
                    SELECT 1 FROM documents 
                    WHERE documents.dossier_id = dossiers.id 
                    AND documents.type_document_id = 16 
                    AND documents.signer = 0
                ) THEN dossiers.id END) as fasigner
            ")
            ->where('dossier_status_id', 2)
            ->where('update', 0)
            ->first();

        // Batch query 3: Devis status counts in ONE query
        $devisCounts = DB::table('dossiers')
            ->join(DB::raw('(
                SELECT dossier_id, devis_status_id, montant_devis,
                       ROW_NUMBER() OVER (PARTITION BY dossier_id ORDER BY created_at DESC) as rn
                FROM devis
            ) as latest_devis'), function($join) {
                $join->on('dossiers.id', '=', 'latest_devis.dossier_id')
                     ->where('latest_devis.rn', '=', 1);
            })
            ->selectRaw("
                COUNT(CASE WHEN latest_devis.devis_status_id = 1 THEN 1 END) as deuxdevis,
                COUNT(CASE WHEN latest_devis.devis_status_id = 2 THEN 1 END) as senddevis,
                COUNT(CASE WHEN latest_devis.devis_status_id = 3 THEN 1 END) as avisdevis,
                COUNT(CASE WHEN latest_devis.devis_status_id = 4 THEN 1 END) as traitdevis_base,
                COUNT(CASE WHEN latest_devis.devis_status_id = 5 THEN 1 END) as count,
                COUNT(CASE WHEN latest_devis.devis_status_id = 9 THEN 1 END) as attdevisav
            ")
            ->where('dossiers.dossier_status_id', 2)
            ->where('dossiers.update', 0)
            ->first();

        // Batch query 4: Facture status counts in ONE query
        $factureCounts = DB::table('dossiers')
            ->join(DB::raw('(
                SELECT dossier_id, devis_status_id as facture_status_id, montant_facture,
                       ROW_NUMBER() OVER (PARTITION BY dossier_id ORDER BY created_at DESC) as rn
                FROM factures
            ) as latest_facture'), function($join) {
                $join->on('dossiers.id', '=', 'latest_facture.dossier_id')
                     ->where('latest_facture.rn', '=', 1);
            })
            ->selectRaw("
                COUNT(CASE WHEN latest_facture.facture_status_id = 1 THEN 1 END) as deuxfacture,
                COUNT(CASE WHEN latest_facture.facture_status_id = 2 THEN 1 END) as sendfacture_base,
                COUNT(CASE WHEN latest_facture.facture_status_id = 3 THEN 1 END) as avisfacture,
                COUNT(CASE WHEN latest_facture.facture_status_id = 4 THEN 1 END) as traitfacture_base,
                COUNT(CASE WHEN latest_facture.facture_status_id = 5 THEN 1 END) as count2,
                COUNT(CASE WHEN latest_facture.facture_status_id = 9 THEN 1 END) as attfactureap
            ")
            ->where('dossiers.dossier_status_id', 2)
            ->where('dossiers.update', 0)
            ->first();

        // Batch query 5: Rapport status counts in ONE query
        $rapportCounts = DB::table('dossiers')
            ->join(DB::raw('(
                SELECT dossier_id, rapport_status_id,
                       ROW_NUMBER() OVER (PARTITION BY dossier_id ORDER BY created_at DESC) as rn
                FROM rapports
            ) as latest_rapport'), function($join) {
                $join->on('dossiers.id', '=', 'latest_rapport.dossier_id')
                     ->where('latest_rapport.rn', '=', 1);
            })
            ->selectRaw("
                COUNT(CASE WHEN latest_rapport.rapport_status_id = 1 THEN 1 END) as deuxrapport,
                COUNT(CASE WHEN latest_rapport.rapport_status_id = 2 THEN 1 END) as sendrapport,
                COUNT(CASE WHEN latest_rapport.rapport_status_id = 4 THEN 1 END) as rapportasigner,
                COUNT(CASE WHEN latest_rapport.rapport_status_id = 5 THEN 1 END) as count3,
                COUNT(CASE WHEN latest_rapport.rapport_status_id = 5 THEN 1 END) as traitrapport
            ")
            ->where('dossiers.dossier_status_id', 2)
            ->where('dossiers.update', 0)
            ->first();

        // Simple counts (fast queries)
        $demandes = Demande::where('status_rendezvous_id', 2)->count();
        $rdv = Rendezvous::where('status_rendezvous_id', 2)->count();

        // Meeting counts
        $meetingCounts = DB::table('dossiers')
            ->selectRaw("
                COUNT(DISTINCT CASE WHEN EXISTS (
                    SELECT 1 FROM meetings 
                    WHERE meetings.dossier_id = dossiers.id 
                    AND meetings.meeting_status_id = 2
                ) THEN dossiers.id END) as enexp,
                COUNT(DISTINCT CASE WHEN EXISTS (
                    SELECT 1 FROM meetings m
                    WHERE m.dossier_id = dossiers.id 
                    AND m.meeting_status_id = 1
                    AND m.created_at = (
                        SELECT MAX(created_at) FROM meetings 
                        WHERE dossier_id = dossiers.id
                    )
                ) AND NOT EXISTS (
                    SELECT 1 FROM rapports WHERE rapports.dossier_id = dossiers.id
                ) THEN dossiers.id END) as areport
            ")
            ->where('dossier_status_id', 2)
            ->where('update', 0)
            ->first();

        // Complex counts that need role-based filtering
        $aexpcount = $this->getAexpCount();
        $attexpap = $this->getAttexpapCount();
        $controldevis = $this->getControlDevisCountFast();
        
        // Role-specific counts
        $traitdevis = $roleId == 2 ? $this->getTraitDevisCountWithSeuil() : $devisCounts->traitdevis_base;
        $traitfacture = $roleId == 2 ? $this->getTraitFactureCountWithSeuil() : $factureCounts->traitfacture_base;

        return [
            // Basic
            'tribunale' => $dossierCounts->tribunale,
            'douteux' => $dossierCounts->douteux,
            'rapide' => $dossierCounts->rapide,
            'demandes' => $demandes,
            'rdv' => $rdv,

            // Documents
            'docasigner' => $docCounts->docasigner,
            'docsigner' => $docCounts->docsigner,
            'basedoc' => $docCounts->basedoc,
            'fasigner' => $docCounts->fasigner,

            // Carence
            'aexpcount' => $aexpcount,
            'attdevis' => $dossierCounts->attdevis,
            'attfacture' => $dossierCounts->attfacture_base,
            'attrapport' => $dossierCounts->attrapport_base,
            'enexp' => $meetingCounts->enexp,
            'areport' => $meetingCounts->areport,
            'attexpap' => $attexpap,

            // Devis
            'count' => $devisCounts->count,
            'deuxdevis' => $devisCounts->deuxdevis,
            'attdevisav' => $devisCounts->attdevisav,
            'traitdevis' => $traitdevis,
            'senddevis' => $devisCounts->senddevis,
            'avisdevis' => $devisCounts->avisdevis,
            'controldevis' => $controldevis,

            // Facture
            'count2' => $factureCounts->count2,
            'deuxfacture' => $factureCounts->deuxfacture,
            'attfactureap' => $factureCounts->attfactureap,
            'traitfacture' => $traitfacture,
            'sendfacture' => $factureCounts->sendfacture_base,
            'avisfacture' => $factureCounts->avisfacture,

            // Rapport
            'count3' => $rapportCounts->count3,
            'deuxrapport' => $rapportCounts->deuxrapport,
            'traitrapport' => $rapportCounts->traitrapport,
            'sendrapport' => $rapportCounts->sendrapport,
            'rapportasigner' => $rapportCounts->rapportasigner,

            // Other
            'dossiersfacture' => $dossierCounts->dossiersfacture,
            'warning' => 0,
        ];
    }

    /**
     * =====================================================
     * HELPER METHODS FOR COMPLEX COUNTS
     * =====================================================
     */

    /**
     * Get count for devis à retraiter
     */
    private function getRetraiterDevisCount()
    {
        $query = $this->getBaseDossiersQuery()->withDevisStatus(5);
        
        if (Auth::user()->role_id == 2) {
            $query->withinSeuil();
        }
        
        return $query->count();
    }

    /**
     * Get count for devis in treatment
     */
    private function getTraitDevisCount()
    {
        $query = $this->getBaseDossiersQuery()->withDevisStatus(4);
        
        if (Auth::user()->role_id == 2) {
            $query->withinSeuil();
        } else {
            $query->where(function ($q) {
                $q->where(function ($subQ) {
                    $subQ->whereIn('mode_id', [2, 3])
                         ->where(function ($valQ) {
                             $valQ->whereNotNull('valeurvenale')
                                  ->orWhereNotNull('valeurepave');
                         });
                })->orWhereNotIn('mode_id', [2, 3]);
            });
        }
        
        return $query->count();
    }

    /**
     * Get count for devis control
     */
    private function getControlDevisCount()
    {
        $query = $this->getBaseDossiersQuery()
            ->with('montant')
            ->withDevisStatus(8);

        if (Auth::user()->role_id == 2) {
            $query->whereHas('montant', function ($q) {
                $q->whereExists(function ($subQuery) {
                    $subQuery->select(DB::raw(1))
                        ->from('seuils')
                        ->whereColumn('seuils.company_id', 'dossiers.company_id')
                        ->where('seuils.user_id', Auth::id())
                        ->whereRaw('montants.montant_accord < seuils.accord');
                });
            });
        } else if (Auth::user()->role_id == 4) {
            // Full access - no additional filter
        }

        return $query->count();
    }

    /**
     * Get count for facture à retraiter
     */
    private function getRetraiterFactureCount()
    {
        $query = $this->getBaseDossiersQuery()->withFactureStatus(5);
        
        if (Auth::user()->role_id == 2) {
            $query->withinSeuil();
        }
        
        return $query->count();
    }

    /**
     * Get count for facture in treatment
     */
    private function getTraitFactureCount()
    {
        $query = $this->getBaseDossiersQuery()->withFactureStatus(4);
        
        if (Auth::user()->role_id == 2) {
            $query->withinSeuil();
        }
        
        return $query->count();
    }

       /**
     * Get count for rapport à retraiter
     */
    private function getRetraiterRapportCount()
    {
        return $this->getBaseDossiersQuery()->withRapportStatus(5)->count();
    }

    /**
     * =====================================================
     * FAST HELPER METHODS FOR COMPLEX COUNTS
     * =====================================================
     */

    /**
     * Get count for dossiers à expertiser (without devis and without doc type 5)
     */
    private function getAexpCount()
    {
        return DB::table('dossiers')
            ->where('dossier_status_id', 2)
            ->where('update', 0)
            ->whereNotExists(function($query) {
                $query->select(DB::raw(1))
                    ->from('devis')
                    ->whereColumn('devis.dossier_id', 'dossiers.id');
            })
            ->whereNotExists(function($query) {
                $query->select(DB::raw(1))
                    ->from('documents')
                    ->whereColumn('documents.dossier_id', 'dossiers.id')
                    ->where('documents.type_document_id', 5);
            })
            ->count();
    }

    /**
     * Get count for dossiers awaiting expertise approval photo
     */
    private function getAttexpapCount()
    {
        return DB::table('dossiers')
            ->where('dossier_status_id', 2)
            ->where('update', 0)
            ->where('type_expertise_id', '!=', 14)
            ->where('mode_id', '!=', 7)
            ->whereNotExists(function($query) {
                $query->select(DB::raw(1))
                    ->from('documents')
                    ->whereColumn('documents.dossier_id', 'dossiers.id')
                    ->where('documents.type_document_id', 6);
            })
            ->whereExists(function($query) {
                $query->select(DB::raw(1))
                    ->from(DB::raw('(
                        SELECT dossier_id, devis_status_id,
                               ROW_NUMBER() OVER (PARTITION BY dossier_id ORDER BY created_at DESC) as rn
                        FROM devis
                    ) as latest_devis'))
                    ->whereColumn('latest_devis.dossier_id', 'dossiers.id')
                    ->where('latest_devis.rn', 1)
                    ->whereIn('latest_devis.devis_status_id', [2, 7]);
            })
            ->whereNotExists(function($query) {
                $query->select(DB::raw(1))
                    ->from('factures')
                    ->whereColumn('factures.dossier_id', 'dossiers.id');
            })
            ->count();
    }

    /**
     * Get count for control devis (fast version)
     */
    private function getControlDevisCountFast()
    {
        $roleId = Auth::user()->role_id;
        
        if ($roleId == 2) {
            return DB::table('dossiers')
                ->join(DB::raw('(
                    SELECT dossier_id, devis_status_id,
                           ROW_NUMBER() OVER (PARTITION BY dossier_id ORDER BY created_at DESC) as rn
                    FROM devis
                ) as latest_devis'), function($join) {
                    $join->on('dossiers.id', '=', 'latest_devis.dossier_id')
                         ->where('latest_devis.rn', 1);
                })
                ->join('montants', 'dossiers.id', '=', 'montants.dossier_id')
                ->whereExists(function($query) {
                    $query->select(DB::raw(1))
                        ->from('seuils')
                        ->whereColumn('seuils.company_id', 'dossiers.company_id')
                        ->where('seuils.user_id', Auth::id())
                        ->whereRaw('montants.montant_accord < seuils.accord');
                })
                ->where('dossiers.dossier_status_id', 2)
                ->where('dossiers.update', 0)
                ->where('latest_devis.devis_status_id', 8)
                ->count();
        } else if ($roleId == 4) {
            return DB::table('dossiers')
                ->join(DB::raw('(
                    SELECT dossier_id, devis_status_id,
                           ROW_NUMBER() OVER (PARTITION BY dossier_id ORDER BY created_at DESC) as rn
                    FROM devis
                ) as latest_devis'), function($join) {
                    $join->on('dossiers.id', '=', 'latest_devis.dossier_id')
                         ->where('latest_devis.rn', 1);
                })
                ->where('dossiers.dossier_status_id', 2)
                ->where('dossiers.update', 0)
                ->where('latest_devis.devis_status_id', 8)
                ->count();
        }
        
        return 0;
    }

    /**
     * Get count for devis in treatment with seuil (role 2)
     */
    private function getTraitDevisCountWithSeuil()
    {
        return DB::table('dossiers')
            ->join(DB::raw('(
                SELECT dossier_id, devis_status_id, montant_devis,
                       ROW_NUMBER() OVER (PARTITION BY dossier_id ORDER BY created_at DESC) as rn
                FROM devis
            ) as latest_devis'), function($join) {
                $join->on('dossiers.id', '=', 'latest_devis.dossier_id')
                     ->where('latest_devis.rn', 1);
            })
            ->whereExists(function($query) {
                $query->select(DB::raw(1))
                    ->from('seuils')
                    ->whereColumn('seuils.company_id', 'dossiers.company_id')
                    ->where('seuils.user_id', Auth::id())
                    ->whereRaw('latest_devis.montant_devis > seuils.min')
                    ->whereRaw('latest_devis.montant_devis < seuils.max');
            })
            ->where('dossiers.dossier_status_id', 2)
            ->where('dossiers.update', 0)
            ->where('latest_devis.devis_status_id', 4)
            ->count();
    }

    /**
     * Get count for facture in treatment with seuil (role 2)
     */
    private function getTraitFactureCountWithSeuil()
    {
        return DB::table('dossiers')
            ->join(DB::raw('(
                SELECT dossier_id, devis_status_id, montant_facture,
                       ROW_NUMBER() OVER (PARTITION BY dossier_id ORDER BY created_at DESC) as rn
                FROM factures
            ) as latest_facture'), function($join) {
                $join->on('dossiers.id', '=', 'latest_facture.dossier_id')
                     ->where('latest_facture.rn', 1);
            })
            ->whereExists(function($query) {
                $query->select(DB::raw(1))
                    ->from('seuils')
                    ->whereColumn('seuils.company_id', 'dossiers.company_id')
                    ->where('seuils.user_id', Auth::id())
                    ->whereRaw('latest_facture.montant_facture > seuils.min')
                    ->whereRaw('latest_facture.montant_facture < seuils.max');
            })
            ->where('dossiers.dossier_status_id', 2)
            ->where('dossiers.update', 0)
            ->where('latest_facture.devis_status_id', 4)
            ->count();
    }
    
    /**
     * =====================================================
     * CARENCE METHODS
     * =====================================================
     */

    public function aexpertiser()
    {
        $this->authorize('attphav', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->withoutDevis()
            ->whereDoesntHave('document', function ($query) {
                $query->where('type_document_id', 5);
            })
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }
    

    public function carencedevis()
    {
        $this->authorize('attdevis', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->withoutDevis()
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function carencefacture()
    {
        $this->authorize('attfacture', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->where('type_expertise_id', '!=', 14)
            ->where('mode_id', '!=', 7)
            ->whereHas('document', function ($query) {
                $query->where('type_document_id', 6);
            })
            ->whereHas('onedevis', function ($query) {
                $query->whereIn('devis_status_id', [2, 7]);
            })
            ->withoutFacture()
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function carencerapport()
    {
        $this->authorize('attrapport', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->whereHas('devis', function ($query) {
                $query->whereIn('devis_status_id', [2, 7]);
            })
            ->whereHas('facture', function ($query) {
                $query->whereIn('devis_status_id', [2, 7]);
            })
            ->withoutRapport()
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function areport()
    {
        $this->authorize('attphav', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->whereHas('onemeeting', function ($query) {
                $query->where('meeting_status_id', 1);
            })
            ->withoutRapport()
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function enexp()
    {
        $this->authorize('enexp', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->whereHas('meeting', function ($query) {
                $query->where('meeting_status_id', 2);
            })
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function attexpap()
    {
        $this->authorize('attphap', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->where('type_expertise_id', '!=', 14)
            ->where('mode_id', '!=', 7)
            ->whereDoesntHave('document', function ($query) {
                $query->where('type_document_id', 6);
            })
            ->whereHas('onedevis', function ($query) {
                $query->whereIn('devis_status_id', [2, 7]);
            })
            ->withoutFacture()
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }
    /**
     * =====================================================
     * DEVIS METHODS
     * =====================================================
     */

    public function devisaretraiter()
    {
        $this->authorize('retraitdevis', Dashboard::class);

        $query = $this->getBaseDossiersQuery()->withDevisStatus(5);

        if (Auth::user()->role_id == 2) {
            $query->withinSeuil();
        }

        $dossiers = $query->limit(self::DEFAULT_LIMIT)->get();
        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function traitdevis()
    {
        $this->authorize('traitdevis', Dashboard::class);

        $query = $this->getBaseDossiersQuery()->withDevisStatus(4);

        if (Auth::user()->role_id == 2) {
            $query->withinSeuil();
        } else {
            $query->where(function ($q) {
                $q->where(function ($subQ) {
                    $subQ->whereIn('mode_id', [2, 3])
                         ->where(function ($valQ) {
                             $valQ->whereNotNull('valeurvenale')
                                  ->orWhereNotNull('valeurepave');
                         });
                })->orWhereNotIn('mode_id', [2, 3]);
            });
        }

        $dossiers = $query->limit(self::DEFAULT_LIMIT)->get();
        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function deuxdevis()
    {
        $this->authorize('att2emedevis', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->withDevisStatus(1)
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function attdevisav()
    {
        $this->authorize('devisattpa', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->withDevisStatus(9)
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function avisdevis()
    {
        $this->authorize('avisdevis', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->withDevisStatus(3)
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function senddevis()
    {
        $this->authorize('devisacc', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->withDevisStatus(2)
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function controldevis()
    {
        $this->authorize('controldevis', Dashboard::class);

        $query = $this->getBaseDossiersQuery()
            ->with('montant')
            ->withDevisStatus(8);

        if (Auth::user()->role_id == 2) {
            $query->whereHas('montant', function ($q) {
                $q->whereExists(function ($subQuery) {
                    $subQuery->select(DB::raw(1))
                        ->from('seuils')
                        ->whereColumn('seuils.company_id', 'dossiers.company_id')
                        ->where('seuils.user_id', Auth::id())
                        ->whereRaw('montants.montant_accord < seuils.accord');
                });
            });
        } else if (Auth::user()->role_id == 4) {
            // Full access - all dossiers
        }

        $dossiers = $query->limit(self::DEFAULT_LIMIT)->get();
        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function devisentraitement()
    {
        $this->authorize('devisattacc', Dashboard::class);

        $query = $this->getBaseDossiersQuery()
            ->withDevisStatus(4)
            ->where(function ($q) {
                $q->where(function ($subQ) {
                    $subQ->whereIn('mode_id', [2, 3])
                         ->where(function ($valQ) {
                             $valQ->whereNull('valeurvenale')
                                  ->whereNull('valeurepave');
                         });
                });
            });

        $dossiers = $query->limit(self::DEFAULT_LIMIT)->get();
        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }
    /**
     * =====================================================
     * FACTURE METHODS
     * =====================================================
     */

    public function facturearetraiter()
    {
        $this->authorize('retraitfacture', Dashboard::class);

        $query = $this->getBaseDossiersQuery()->withFactureStatus(5);

        if (Auth::user()->role_id == 2) {
            $query->withinSeuil();
        }

        $dossiers = $query->limit(self::DEFAULT_LIMIT)->get();
        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function traitfacture()
    {
        $this->authorize('traitfacture', Dashboard::class);

        $query = $this->getBaseDossiersQuery()->withFactureStatus(4);

        if (Auth::user()->role_id == 2) {
            $query->withinSeuil();
        }

        $dossiers = $query->limit(self::DEFAULT_LIMIT)->get();
        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function deuxfacture()
    {
        $this->authorize('att2emefacture', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->withFactureStatus(1)
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function attfactureap()
    {
        $this->authorize('factureattph', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->withFactureStatus(9)
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function avisfacture()
    {
        $this->authorize('avisfacture', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->withFactureStatus(3)
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function sendfacture()
    {
        $this->authorize('factureacc', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->whereDoesntHave('rapport')
            ->withFactureStatus(2)
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function factureentraitement()
    {
        $this->authorize('factureattacc', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->withFactureStatus(4)
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }
    /**
     * =====================================================
     * RAPPORT METHODS
     * =====================================================
     */

    public function rapportaretraiter()
    {
        $this->authorize('instacerapport', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->withRapportStatus(5)
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function traitrapport()
    {
        $this->authorize('instacerapport', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->withRapportStatus(5)
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function rapportasigner()
    {
        $this->authorize('rappsigner', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->withRapportStatus(4)
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function sendrapport()
    {
        $this->authorize('rapportenv', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->withRapportStatus(2)
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function deuxrapport()
    {
        $this->authorize('rapportref', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->withRapportStatus(1)
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    /**
     * =====================================================
     * OTHER FILTERS & SPECIAL CATEGORIES
     * =====================================================
     */

    public function douteux()
    {
        $this->authorize('douteux', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->douteux()
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function tribunale()
    {
        $this->authorize('tribunale', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->tribunale()
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function rapide()
    {
        $this->authorize('rapide', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->rapide(Auth::user()->role_id)
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function warning()
    {
        $this->authorize('attphav', Dashboard::class);

        // Get dossiers that exceed time limits
        $dossiers = $this->getBaseDossiersQuery()
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        // Attach timeleft
        $this->attachTimeleft($dossiers);

        // Filter only overdue dossiers (timeleft < 0)
        $overdueDossiers = $dossiers->filter(function ($dossier) {
            return $dossier->timeleft < 0;
        })->values();

        return response()->json($overdueDossiers);
    }

    public function docasigner()
    {
        $this->authorize('docasigner', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->withDocumentsToSign(21)
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function docsigner()
    {
        $this->authorize('docasigner', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->withSignedDocuments(21)
            ->whereDoesntHave('onerapport', function ($query) {
                $query->whereIn('rapport_status_id', [2, 3]);
            })
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function fasigner()
    {
        $this->authorize('afacturé', Dashboard::class);

        $dossiers = Dossier::where('dossier_status_id', 2)
            ->where('update', 0)
            ->select([
                'id',
                'ref',
                'company_id',
                'insured_id',
                'matricule',
                'new_matricule',
                'numero_sinistre',
                'type_expertise_id',
                'mode_id',
                'step_id',
                'dossier_status_id',
            ])
            ->with([
                'company:id,company',
                'onefacture:id,dossier_id,montant_facture,devis_status_id',
                'onedevis:id,dossier_id,montant_devis,devis_status_id',
                'mode:id,mode',
                'currentStep:id,step',
                'insured:id,name',
                'typeexpertise:id,type',
                'status:id,status,color'
            ])
            ->whereHas('document', function ($query) {
                $query->where('type_document_id', 16)->where('signer', 0);
            })
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function basedoc()
    {
        $this->authorize('attdocbase', Dashboard::class);

        $dossiers = $this->getBaseDossiersQuery()
            ->withoutBaseDocuments()
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }

    public function facturation()
    {
        $this->authorize('afacturé', Dashboard::class);

        $dossiers = Dossier::select([
                'id',
                'ref',
                'company_id',
                'insured_id',
                'matricule',
                'new_matricule',
                'numero_sinistre',
                'type_expertise_id',
                'mode_id',
                'step_id',
                'dossier_status_id',
                'facture',
                'update'
            ])
            ->with([
                'company:id,company',
                'onefacture:id,dossier_id,montant_facture,devis_status_id',
                'onedevis:id,dossier_id,montant_devis,devis_status_id',
                'mode:id,mode',
                'currentStep:id,step',
                'insured:id,name',
                'typeexpertise:id,type',
                'status:id,status,color'
            ])
            ->whereNull('facture')
            ->where('dossier_status_id', 2)
            ->where('update', 1)
            ->limit(self::DEFAULT_LIMIT)
            ->get();

        $this->attachTimeleft($dossiers);

        return response()->json($dossiers);
    }
    
    /**
     * =====================================================
     * CACHE MANAGEMENT
     * =====================================================
     */

    /**
     * Clear dashboard cache for current user
     */
    public function clearCache()
    {
        $userId = Auth::id();
        $roleId = Auth::user()->role_id;
        $cacheKey = "dashboard_front_{$userId}_{$roleId}";

        Cache::forget($cacheKey);

        return response()->json(['message' => 'Cache cleared successfully']);
    }

    /**
     * Clear all dashboard caches (admin only)
     */
    public function clearAllCaches()
    {
        $this->authorize('fullaccess', Dashboard::class);

        Cache::flush();

        return response()->json(['message' => 'All caches cleared']);
    }
}